<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Main Konta Core Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */

final class Konta_Extension {

	/**
	 * Plugin Version
	 *
	 * @since 1.0.0
	 *
	 * @var string The plugin version.
	 */
	const VERSION = '1.0.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */

	const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.0';


	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Elementor_Test_Extension The single instance of the class.
	 */

	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Elementor_Test_Extension An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {
		add_action( 'plugins_loaded', [ $this, 'init' ] );
	}

	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {

		// Check if Elementor installed and activated

		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return;
		}

		// Check for required Elementor version

		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return;
		}

		// Check for required PHP version

		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return;
		}


		// Add Plugin actions

		add_action( 'elementor/widgets/register', [ $this, 'init_widgets' ] );


        // Register widget scripts

		add_action( 'elementor/frontend/after_enqueue_scripts', [ $this, 'widget_scripts' ]);


		// Specific Register widget scripts

		// add_action( 'elementor/frontend/after_register_scripts', [ $this, 'konta_regsiter_widget_scripts' ] );
		// add_action( 'elementor/frontend/before_register_scripts', [ $this, 'konta_regsiter_widget_scripts' ] );


        // category register

		add_action( 'elementor/elements/categories_registered',[ $this, 'konta_elementor_widget_categories' ] );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'konta' ),
			'<strong>' . esc_html__( 'Konta Core', 'konta' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'konta' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */

			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'konta' ),
			'<strong>' . esc_html__( 'Konta Core', 'konta' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'konta' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}
	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(

			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'konta' ),
			'<strong>' . esc_html__( 'Konta Core', 'konta' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'konta' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */

	public function init_widgets() {

		$widget_register = \Elementor\Plugin::instance()->widgets_manager;

		// Header Include file & Widget Register
		require_once( KONTA_ADDONS . '/header/header.php' );
		require_once( KONTA_ADDONS . '/header/header-v2.php' );
		require_once( KONTA_ADDONS . '/header/header-v3.php' );
		require_once( KONTA_ADDONS . '/header/header-v4.php' );


		$widget_register->register ( new \Konta_Header() );
		$widget_register->register ( new \Konta_Header2() );
		$widget_register->register ( new \Konta_Header_3() );
		$widget_register->register ( new \Konta_Header_v4() );

		// Include All Widget Files
		foreach($this->Konta_Include_File() as $widget_file_name){
			require_once( KONTA_ADDONS . '/widgets/konta-'."$widget_file_name".'.php' );
		}

		// Include All new version Widget Files
		foreach($this->Konta_Include_File_New() as $widget_file_name){
			require_once( KONTA_ADDONS . '/widgets/konta-new/konta-'."$widget_file_name".'.php' );
		}

		// All Widget Register
		foreach($this->Konta_Register_File() as $name){
			$widget_register->register ( $name );
		}
		
	}

	public function Konta_Include_File(){
		return [
			'banner', 
			'button', 
			'section-title', 
			'animated-shape', 
			'menu-select', 
			'service', 
			'advane-tab', 
			'group-image', 
			'features', 
			'image-box', 
			'counterup', 
			'project', 
			'wcu', 
			'team', 
			'testimonials', 
			'brand', 
			'blog', 
			'btn-with-author', 
			'faq', 
			'about-us-widget', 
			'workprocess', 
			'contact-form', 
			'pricing', 
			'image-with-video', 
			'what-we-do', 
			'newslater', 
			'cta', 
			'download-file', 
			'gallery', 
			'contact-info', 
			'skill', 

		];
	}

	public function Konta_Include_File_New(){
		return [
			'about-us', 
			'animated-shape', 
			'banner', 
			'banner2', 
			'blog', 
			'brand-logo', 
			'button', 
			'contact-info', 
			'contact-us', 
			'counterup', 
			'download-button-v2', 
			'faq', 
			'features', 
			'gallery', 
			'group-image', 
			'image-box', 
			'image-tab', 
			'image-with-video', 
			'misson', 
			'newslatter-form', 
			'pricing', 
			'progressbar', 
			'project', 
			'project-info', 
			'section-title', 
			'services', 
			'slider-arrow', 
			'team', 
			'testimonial', 
			'video-button', 
			'workprocess', 
			'why-chose-us', 
			

		];
	}

	public function Konta_Register_File(){
		return [
			new \Konta_Banner() ,
			new \Konta_Animated_Shape(),
			new \Konta_Button(),
			new \Kunta_Menu(),
			new \Konta_Section_Title_Widget(),
			new \Konta_Service(),
			new \Konta_Tab_Builder(),
			new \Konta_Group_Image(),
			new \Konta_Feature(),
			new \Konta_Image(),
			new \Konta_Counterup(),
			new \Konta_Project(),
			new \Konta_WCU(),
			new \Konta_Team(),
			new \Konta_Testimonials(),
			new \Konta_Brand(),
			new \Konta_Blog_Post(),
			new \Konta_Button_Author(),
			new \Konta_Faq(),
			new \Konta_A_S_W(),
			new \Konta_Work_Process(),
			new \Konta_Contact_Form(),
			new \Konta_Pricing(),
			new \Konta_Video_Thumb(),
			new \Konta_Work(),
			new \Konta_Newslatter_Form(),
			new \Konta_CTA(),
			new \Konta_Download_Button_Box(),
			new \konta_Gallery(),
			new \Konta_Contact_Info_V2(),
			new \konta_Skill(),





			
			new \Konta_About_Us_2(),
			new \Konta_Animated_Image_2(),
			new \Konta_Banner_2(),
			new \Konta_Banner2_2(),
			new \Konta_Blog_Post_2(),
			new \Konta_Client_Logo_2(),
			new \Konta_Button_2(),
			new \Konta_Contact_Info_2(),
			new \Konta_Contact_2(),
			new \Konta_Counterup_2(),
			new \Konta_Download_Box_2(),
			new \Konta_Faq_2(),
			new \Konta_Feature_2(),
			new \Konta_Gallery_2(),
			new \Konta_Group_Image_2(),
			new \Konta_Image_2(),
			new \Konta_Image_Tab_Box_2(),
			new \Konta_Video_Thumb_2(),
			new \Konta_Mission_Box_2(),
			new \Konta_Newslatter_Form_2(),
			new \Konta_Pricing_2(),
			new \Konta_Skill_Box_2(),
			new \Konta_Project_Info_Widget_2(),
			new \Konta_Project_2(),
			new \Konta_Section_Title_Widget_2(),
			new \Konta_Service_2(),
			new \Konta_Slider_Arrow_2(),
			new \Konta_Team_2(),
			new \Konta_Testimonial_Slider_2(),
			new \Konta_Play_Button_2(),
			new \Konta_WorkProcess_Box_2(),
			new \Konta_Why_Chose_Us(),


			
		];
	}

    public function widget_scripts() {

        wp_enqueue_script(
            'konta-frontend-script',
            KONTA_PLUGDIRURI . 'assets/js/konta-frontend.js',
            array('jquery'),
            false,
            true
		);

	}


	// public function konta_regsiter_widget_scripts( ) {

	// 	wp_register_script(
 //            'konta-tilt',
 //            KONTA_PLUGDIRURI . 'assets/js/tilt.jquery.min.js',
 //            array('jquery'),
 //            false,
 //            true
	// 	);
	// }


    function konta_elementor_widget_categories( $elements_manager ) {

        $elements_manager->add_category(
            'konta',
            [
                'title' => __( 'Konta', 'konta' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );

        $elements_manager->add_category(
            'konta_footer_elements',
            [
                'title' => __( 'Konta Footer Elements', 'konta' ),
                'icon' 	=> 'fa fa-plug',
            ]
		);

		$elements_manager->add_category(
            'konta_header_elements',
            [
                'title' => __( 'Konta Header Elements', 'konta' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );
	}
}

Konta_Extension::instance();